/***************************************************************************
 *
 * Copyright (C) 2001 International Business Machines
 * All rights reserved.
 *
 * This file is part of the GPFS mmfslinux kernel module.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer. 
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution. 
 *  3. The name of the author may not be used to endorse or promote products 
 *     derived from this software without specific prior written
 *     permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *************************************************************************** */
//#define TEST_LOCK_MOVE_CLAMP

#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/fcntl.h>
#include <linux/slab.h>
#include <linux/file.h>
#include <linux/smp_lock.h>
#include <sys/syscall.h>
#include <asm/uaccess.h>

#ifndef __KERNEL__
#define __KERNEL__
#endif
#include <oplock.h>
#include <Trace.h>
#include <cxiTypes.h>

#if (LINUX_VERSION_CODE < 0x02032b)
struct wait_queue *oplock_queue = NULL;
#else
DECLARE_WAIT_QUEUE_HEAD(oplock_queue);
#endif

int setSMBOplock(int fileDesc, int accessWant, int oplockWant,
                 void *breakArgP, int *oplockGrantedP);

int cxiSMBOpenLockMove(struct file *filp, int tokenop, int lockmode, int nodenumber);

static void dump_oplock_entry(fd, cmd, arg)
{
  static char *cmdstr, *omodestr, *dmodestr;

  switch (cmd & 0xff) {
  case F_OPLKREG: cmdstr = "F_OPLKREG"; break;
  case F_OPLKREG_LVL2: cmdstr = "F_OPLKREG_LVL2"; break;
  case F_OPLKACK: cmdstr = "F_OPLKACK"; break;
  case F_OPLKSTAT: cmdstr = "F_OPLKSTAT"; break;
  case F_SET_SHMODE: cmdstr = "F_SET_SHMODE"; break;
  case F_DEL_SHMODE: cmdstr = "F_DEL_SHMODE"; break;
  case F_OPENMODE: cmdstr = "F_OPENMODE"; break;
  case F_SET_NLM_MODE: cmdstr = "F_SET_NLM_MODE"; break;
  case F_DEL_NLM_MODE: cmdstr = "F_DEL_NLM_MODE"; break;
  case F_SHARE_MODE: cmdstr = "F_SHARE_MODE"; break;
  case F_EXCL_OPEN: cmdstr = "F_EXCL_OPEN"; break;
  case F_SHARED_OPEN: cmdstr = "F_SHARED_OPEN"; break;
  default: cmdstr = "F_UNKNOWN";
  }

  switch ((cmd >> 8) & 0xff) {
  case 0: omodestr = "RDONLY"; break;
  case 1: omodestr = "WRONLY"; break;
  case 2: omodestr = "RDWR"; break;
  case 4: omodestr = "APPEND"; break;
  case 0xf: omodestr = "FCB"; break;
  default: omodestr = "OMODE_UNKNOWN";
  }

  switch (arg & 0xff) {
  case DENY_DOS: dmodestr = "DENY_DOS"; break;
  case DENY_ALL: dmodestr = "DENY_ALL"; break;
  case DENY_WRITE: dmodestr = "DENY_WRITE"; break;
  case DENY_READ: dmodestr = "DENY_READ"; break;
  case DENY_NONE: dmodestr = "DENY_NONE"; break;
  case DENY_FCB: dmodestr = "DENY_FCB"; break;
  default: dmodestr = "DENY_UNKNOWN";
  }
  TRACE4(TRACE_SMB, 5, TRCID_DUMP_OPLOCK,
         "oplock_fcntl: %s on %d (%s %s)\n",
         cmdstr, fd, omodestr, dmodestr);
}

/* At entry, parameters are different for this type of lock request.
 * Because the NLM does a system-style open in which it provides the
 * file structure and it is never in a file descriptor list as for
 * normal users, the "file" argument is the address of the file
 * structure, passed in the third (arg) parameter.  The command
 * code is passed in the low order 8 bits of cmd, the access mode
 * in the next 8 bits, and the deny modes in the third 8 bits. */
static int nlm_mode(int cmd, int arg)
{
  int lockmode = 0;
  int accessmode = 0;
  int denymode = 0;
  struct file *filp;
  int tokenop;
  int err;

  accessmode = (cmd >> 8) & 0xff;
  denymode = (cmd >> 16) & 0xff;
  filp = (struct file *)arg;
  if (filp == NULL)
    return -EINVAL;

  /* Translate NLM access and deny modes into those used within GPFS */
  if (accessmode & NLM_READ)
    lockmode = coRead;
  if (accessmode & NLM_WRITE)
    lockmode += (coWriteM + coWriteA);
  if (denymode & NLM_READ)
    lockmode += coDenyR;
  if (denymode & NLM_WRITE)
    lockmode += coDenyWA;

  if ((cmd & 0xff) == F_SET_NLM_MODE)
  {
    tokenop = TOKENOP_ACQUIRE_NLM;
    TRACE1(TRACE_SMB, 5, TRCID_NLM_SET,
           "nlm_mode: Lock acquire to GPFS is 0x%x\n", lockmode);
  }
  else
  {
    tokenop = TOKENOP_RELEASE_NLM;
    TRACE1(TRACE_SMB, 5, TRCID_NLM_RELEASE,
           "nlm_mode: Lock release to GPFS is 0x%x\n", lockmode);
  }

  if (lockmode > 0)
    err = cxiSMBOpenLockControl(filp, tokenop, lockmode);
  else
    err = 0;

  TRACE1(TRACE_SMB, 9, TRCID_NLM_EXIT, "nlm_mode: GPFS request complete.  RC=%d\n",err);

  return err;
}



/* This flavor of NLM request is to move locks from one system to
 * another.  The additional parameter is the node number of the
 * system from which locks may be taken. This flavor passes this
 * node number along but is in other respects identical to the
 * standard NLM lock requests. */

/* At entry, parameters are different for this type of lock request.
 * Because the NLM does a system-style open in which it provides the
 * file structure and it is never in a file descriptor list as for
 * normal users, the "file" argument is the address of the file
 * structure, passed in the third (arg) parameter.  The command
 * code is passed in the low order 8 bits of cmd, the access mode
 * in the next 8 bits, and the deny modes in the third 8 bits. */
static int nlm_move(int cmd, int arg, int nodenumber)
{
  int lockmode = 0;
  int accessmode = 0;
  int denymode = 0;
  struct file *filp;
  int tokenop;
  int err;

  accessmode = (cmd >> 8) & 0xff;
  denymode = (cmd >> 16) & 0xff;
  filp = (struct file *)arg;
  if (filp == NULL)
    return -EINVAL;

  /* Translate NLM access and deny modes into those used within GPFS */
  if (accessmode & NLM_READ)
    lockmode = coRead;
  if (accessmode & NLM_WRITE)
    lockmode += (coWriteM + coWriteA);
  if (denymode & NLM_READ)
    lockmode += coDenyR;
  if (denymode & NLM_WRITE)
    lockmode += coDenyWA;

  tokenop = TOKENOP_ACQUIRE_NLM;
  TRACE2(TRACE_SMB, 5, TRCID_NLM_MOVE,
         "nlm_move: Lock acquire(move) to GPFS is 0x%x, source node %d\n",
         lockmode, nodenumber);

  if (lockmode > 0)
    err = cxiSMBOpenLockMove(filp, tokenop, lockmode, nodenumber);
  else
    err = 0;

  TRACE1(TRACE_SMB, 9, TRCID_NLM_MOVE_EXIT, "nlm_move: GPFS request complete.  RC=%d\n",err);

  return err;
}


#ifdef TEST_LOCK_MOVE_CLAMP
/* The following routines are test drivers for functions provided to the NLM for
 * lock moving and to recovery for lock clamping */

#include <cxiSharedSeg-plat.h>
#include <linux/fs.h>
#include <linux/fcntl.h>

int test_move_open_lock(unsigned int fd, unsigned int cmd, unsigned long arg)
{
  struct file *filp;
  int rc;

  TRACE3(TRACE_SMB, 1, TRCID_TESTMOVEOPENLOCK,
         "test_move_open_lock: fd: 0x%X, cmd 0x%X, arg: 0x%X",
         fd, cmd, arg);

  filp = fget(fd);

  rc = nlm_move(cmd, (int)filp, arg);

  fput(filp);
  return rc;
}

int
gpfs_f_lock(struct file *fP, int cmd, struct file_lock *flP);

int test_move_rec_lock(unsigned int fd, unsigned int cmd, unsigned long arg)
{
  struct file *filp;
  struct file_lock fl;
  int rc;

  TRACE3(TRACE_SMB, 1, TRCID_TESTMOVERECLOCK,
         "test_move_rec_lock: fd: 0x%X, cmd 0x%X, arg: 0x%X",
         fd, cmd, arg);

  filp = fget(fd);
  fl.fl_next = NULL;
  fl.fl_link.next = NULL;
  fl.fl_link.prev = NULL;
  fl.fl_block.next = NULL;
  fl.fl_block.prev = NULL;
  fl.fl_owner = cxiGetFcntlOwner(NULL);
  fl.fl_pid = current->pid;
  // fl.fl_wait = NULL;
  fl.fl_file = filp;
  fl.fl_flags = 0;
  fl.fl_type = F_WRLCK;
  fl.fl_start = 0;
  fl.fl_end = 10;

  rc = gpfs_f_lock(filp, F_SETLK | 0x8000 | arg<<16, &fl);

  fput(filp);
  return rc;
}

/* Vector table for all routines that can be called with the ss_fs_ioctl. */
int (*ss_ioctl_op[MAX_SS_IOCTL_OPS+1])();

int test_clamp_locks(unsigned int fd, int clamp)
{
  struct file *filp;
  struct inode *inode;
  int rc;

  TRACE2(TRACE_SMB, 1, TRCID_TESTCLAMPLOCKS,
         "test_clamp_locks: fd 0x%X, clamp %d",
         fd, clamp);

  filp = fget(fd);
  inode = filp->f_dentry->d_inode;

  if (clamp<2)
    if (ss_ioctl_op[ClampLocks] == NULL)
      rc = 22;
    else
      if (clamp==1)
        rc = ss_ioctl_op[ClampLocks](NULL);
      else
        rc = ss_ioctl_op[ClampLocks](inode->i_sb);
  else
    if (ss_ioctl_op[UnClampLocks] == NULL)
      rc = 23;
    else
      if (clamp==3)
        rc = ss_ioctl_op[UnClampLocks](NULL);
      else
        rc = ss_ioctl_op[UnClampLocks](inode->i_sb);

  fput(filp);
  return rc;
}

#endif

/* The F_SET_SHMODE operation requests a specific share mode.
   It is invoked by both CIFS and NLM to get clearance for their type
   of access. The parameter arg includes the DENY_MODE bits as well as
   indicator flags specifying Read/Write access and the origin system
   (Samba or NLM).

   The result of the call may be that the request is granted or
   that the request is denied.  All requests are either satisfied
   immediately or denied. There is no waiting. */
static int set_share_mode(struct file *filp, __u32 deny_type,
                          int omode, int allow_share_delete)
{
  /* First, check if we have local permission for this request */
  /* The rules are as follows:                                 */
  /*   DENY_READ   requires  coDenyR                           */
  /*   DENY_WRITE  requires  coDenyWM  coming from CIFS        */
  /*   DENY_WRITE  requires  coDenyWA  coming from NLM         */
  /*   DENY_FCB    requires  coDenyR and                       */
  /*                         coDenyWM                          */
  /*   DENY_DOS    requires  coDenyR and                       */
  /*                         coDenyWM                          */
  /*   DENY_ALL    requires  coDenyR and                       */
  /*                         coDenyWM                          */
  /*   Reading     requires  coRead    coming from CIFS        */
  /*   Writing     requires  coWriteM and                      */
  /*                         coWriteA  coming from CIFS        */
  /*   Writing     requires  coWriteM  coming from NFS/NLM     */
  /*                                                           */
  int token_want = 0;
  int err = 0;

  switch(deny_type)   {
  case DENY_DOS:      /* Treat like DENY_WRITE for non-executables */
  case DENY_WRITE:
    token_want |= (coDenyWM | coDenyWA);
    break;
  case DENY_READ:
    token_want |= coDenyR;
    break;
  case DENY_FCB:
  case DENY_ALL:
    token_want |= (coDenyR | coDenyWM | coDenyWA);
  case DENY_NONE:
    break;
  default:
    err = EINVAL;
    goto xerror;
  }

  switch (omode) {
  case 0: /* DOS_OPEN_RDONLY */
    token_want |= coRead;
    break;
  case 1: /* DOS_OPEN_WRONLY */
  case 4: /* DOS_OPEN_APPEND */
    token_want |= (coWriteM | coWriteA);
    break;
  case 2: /* DOS_OPEN_RDWR */
  case 0xf: /* DOS_OPEN_FCB */
    token_want |= (coRead | coWriteM | coWriteA);
    break;
  default:
    err = EINVAL;
    goto xerror;
  }

  token_want |= allow_share_delete;

    /* If some lock bits released, notify GPFS */
  TRACE1(TRACE_SMB, 5, TRCID_FCNTL_TOKEN,
         "  Lock acquire to GPFS is: 0x%x\n",
         token_want);

  err = cxiSMBOpenLockControl(filp, TOKENOP_ACQUIRE, token_want);

  TRACE1(TRACE_SMB, 9, TRCID_FCNTL_SET,
         "GPFS request complete.  RC=%d\n",err);
 xerror:
  return -err;
}

/* Relinquish a share mode.
   This function is always successful. Identify the token bits
   corresponding to the access modes and deny modes.
   Send message to GPFS to release them. */
static int delete_share_mode(struct file *filp, __u32 deny_type,
                             int omode, int allow_share_delete)
{
  int token_want = 0;
  int err = 0;

  switch(deny_type) {
  case DENY_DOS:      /* Treat like DENY_WRITE for non-executables */
  case DENY_WRITE:
    token_want |= (coDenyWM | coDenyWA);
    break;
  case DENY_READ:
    token_want |= coDenyR;
    break;
  case DENY_FCB:
  case DENY_ALL:
    token_want |= (coDenyR | coDenyWM | coDenyWA);
  case DENY_NONE:
    break;
  default:
    err = EINVAL;
    goto xerror;
  }

  switch (omode) {
  case 0: /* DOS_OPEN_RDONLY */
    token_want |= coRead;
    break;
  case 1: /* DOS_OPEN_WRONLY */
  case 4: /* DOS_OPEN_APPEND */
    token_want |= (coWriteM | coWriteA);
    break;
  case 2: /* DOS_OPEN_RDWR */
  case 0xf: /* DOS_OPEN_FCB */
    token_want |= (coRead | coWriteM | coWriteA);
    break;
  default:
    err = EINVAL;
    goto xerror;
  }

  token_want |= allow_share_delete;

  /* If some lock bits released, notify GPFS */
  TRACE1(TRACE_SMB, 5, TRCID_FCNTL_RELEASE,
         "  Lock release to GPFS is: 0x%x\n",
         token_want);

  cxiSMBOpenLockControl(filp, TOKENOP_RELEASE, token_want);

  TRACE5(TRACE_SMB, 5, TRCID_FCNTL_TOKEN_RELEASE,
         "sys_fcntl: del share mode %d, omode %d for file %x:%ld (%s)\n",
         deny_type, omode,
         filp->f_dentry->d_inode->i_dev, filp->f_dentry->d_inode->i_ino,
         filp->f_dentry->d_name.name);
 xerror:
  return -err;
}

/*
 * Samba implements oplock support in the kernel by opening
 * a pipe into the kernel to receive kernel oplock break
 * messages. Typically Samba will register an oplock with
 * F_OPLKREG and break it with F_OPLKACK
 *
 */

/* Parameters in are the following:
 *      fd       file descriptor
 *      cmd      0:7 bits is command (R_OPLKREG, ...)
 *               (lockd) 8:15 bits are access mode (NFS_READ, NFS_WRITE)
 *               (Samba) 15 bit is ALLOW_SHARE_DELETE
                 (lockd) 16:23 bits are deny mode (NFS_READ, NFS_WRITE)
                 (Samba) 16:23 bits are deny mode (DENY_READ, ...)
 *      arg      (Samba) pipe descriptor only valid on oplock calls
 *               (lockd) value is file structure address
 */

int oplock_fcntl(unsigned int fd, unsigned int cmd, unsigned long arg)
{
  struct file *filp = NULL, *pfilp = NULL;
  struct inode *inode;
  long err = -EBADF;
  long err_oplock;
  __u32 deny_type;
  int reading;
  int writing;
  int src_cifs;
  int token_want;
  int omode, optype;
  int allow_share_delete;
  int grant;
  int oplockWant;
  int pfd;

  dump_oplock_entry(fd, cmd, arg);

  filp = fget(fd);
  if (!filp) {
    TRACE0(TRACE_SMB, 1, TRCID_FCNTL_FILP, "OpLock: file pointer was NULL");
    goto out;
  }
  inode = (struct inode *)filp->f_dentry->d_inode;

  optype = cmd & 0xff;
  omode = (cmd >> 8) & 0xf;
  deny_type = (cmd >> 16) & DENY_MASK;
  allow_share_delete = cmd & ALLOW_SHARE_DELETE;

  switch (optype) {
  case F_SHARE_MODE:
    err = set_share_mode(filp, deny_type, omode, allow_share_delete);
    break;

    /*
     * XXXX SET_SHMODE, DEL_SHMODE, F_OPENMODE are deprecated and should not
     * be used. They are currently here only for backward compatibility with
     * old versions of samba
     */
  case F_SET_SHMODE:
    deny_type = (__u32)arg & DENY_MASK;
    allow_share_delete = arg & ALLOW_SHARE_DELETE;
    err = set_share_mode(filp, deny_type, omode, allow_share_delete);
    break;

  case F_DEL_SHMODE:
    deny_type = (__u32)arg & DENY_MASK;
    allow_share_delete = arg & ALLOW_SHARE_DELETE;
    err = delete_share_mode(filp, deny_type, omode, allow_share_delete);
    break;

   case F_OPENMODE:
     pfd = (int)arg;
     pfilp = fget(pfd);
     if (!pfilp) goto out_putf;

     err = setSMBOplock(fd, opAccessRead, -1 /* no change */, (void *)pfilp, &grant);
     if (!err) /* grant check ? */
       TRACE2(TRACE_SMB, 9, TRCID_FCNTL_OPENMODE_OK,
             "sys_fcntl: open mode registered on file:"
             "dev = %x, inode = %ld\n",
             inode->i_dev, inode->i_ino);
     else {
       TRACE4(TRACE_SMB, 4, TRCID_FCNTL_OPENMODE_DENIED,
              "sys_fcntl: registering open mode failed on file:"
              "dev = %x, inode = %ld (rc %d, grant %d)\n",
              inode->i_dev, inode->i_ino, err, grant);
       if (!err) err = -EAGAIN;
     }
     break;

  case F_OPLKREG:
  case F_OPLKREG_LVL2:
  case F_SHARED_OPEN:
  case F_EXCL_OPEN:
      /* This is a combined operation of getting oplocks and share mode locks.
       * The return values generated here are the followoing:
       *    EOK:     everything was successful
       *    EACCES:  the share mode request was rejected
       *    E2BIG:   oplock request for exclusive, share was granted
       *    EAGAIN:  no oplock was granted
       *    other:   something horrible happened  */
    err_oplock = 0;
    pfd = (int)arg;
    pfilp = fget(pfd);
    if (!pfilp) goto out_putf;

    if (optype == F_OPLKREG || optype == F_EXCL_OPEN)
      oplockWant = smbOplockExclusive;
    else
      oplockWant = smbOplockShared;

    err = setSMBOplock(fd, opAccessRead, oplockWant, (void *)pfilp, &grant);
    if (err != 0)
      break;

    if (grant == 0)
      err_oplock = -EAGAIN; /* refused oplock */

    /* If exclusive oplock was requested and only shared is available,
     * return error code E2BIG to indicate this
     */
    if (oplockWant != grant)
        err_oplock = -E2BIG;

    TRACE5(TRACE_SMB, 9, TRCID_FCNTL_OPLOCK_SHARED,
           "sys_fcntl: granted oplock type %d on file:"
           "dev = %x, inode = %ld; filp %lx, pfilp %lx\n",
           grant, inode->i_dev, inode->i_ino,
           (unsigned long)filp,
           (unsigned long)pfilp);

    if (optype == F_SHARED_OPEN || optype == F_EXCL_OPEN) {
      err = set_share_mode(filp, deny_type, omode, allow_share_delete);
      if (err != 0 && err_oplock != -EAGAIN)   /* If an oplock was granted, release it */
        setSMBOplock(fd, opAccessNone, smbOplockNone, (void *)pfilp, &grant);
    }

    if (err == 0)  /* If no share mode error */
      err = err_oplock;
    break;

  case F_OPLKACK:
    switch (arg) {
    case OP_REVOKE:
    case OP_DOWNGRADE:
      if (arg == OP_REVOKE)
        oplockWant = smbOplockNone;
      else
        oplockWant = smbOplockShared;

      err = setSMBOplock(fd, -1, oplockWant, NULL, &grant);
      if (err != 0)
        goto wakeup;

      if (grant == oplockWant)
        TRACE2(TRACE_SMB, 9, TRCID_FCNTL_OPLKACK_OK,
               "sys_fcntl: revoked kernel oplock on file: dev = %x, inode = %ld\n",
               inode->i_dev, inode->i_ino);
      else {
        TRACE4(TRACE_SMB, 4, TRCID_FCNTL_OPLKACK_FAIL,
               "sys_fcntl: revoke oplock failed on file: dev = %x, inode = %ld (grant %d, err %d)\n",
               inode->i_dev, inode->i_ino, grant, err);
        err = -EAGAIN;
      }
    wakeup:
      wake_up_interruptible(&oplock_queue);
      break;

    case -1: /* XXXX return state */
      err = -1; /* return state of 1 */
      break;

    default:
      err = -EINVAL;
      break;
    }
    break;

  case F_OPLKSTAT: {
    oplock_stat_t *os = (oplock_stat_t *)arg;
    err = filp->f_op->read(filp, (void *)os, sizeof(oplock_stat_t), &filp->f_pos);
    if (err != sizeof(oplock_stat_t)) {
      TRACE1(TRACE_SMB, 4, TRCID_FCNTL_STAT,
             "sys_fcntl: read on filp failed; bytes read %ld\n", err);
      err = -err;
    }
    else
      err = 0;
    break;
  }
  }
 out_putp:
  if (pfilp) fput(pfilp);
 out_putf:
  if (filp) fput(filp);
 out:
  return err;
}

int newsys_fcntl(unsigned int fd, unsigned int cmd, unsigned long arg)
{
  switch (cmd & 0xff) {
  case F_OPLKREG:
  case F_OPLKREG_LVL2:
  case F_OPLKACK:
  case F_OPLKSTAT:
  case F_SET_SHMODE:
  case F_DEL_SHMODE:
  case F_SHARE_MODE:
  case F_EXCL_OPEN:
  case F_SHARED_OPEN:
    return oplock_fcntl(fd, cmd, arg);
  case F_SET_NLM_MODE:
  case F_DEL_NLM_MODE:
    return nlm_mode(cmd, arg);
  case F_SET_NLM_MOVE:
    return nlm_move(cmd, arg, fd);

#ifdef TEST_LOCK_MOVE_CLAMP
  case F_TEST_MOVE_OPEN_LOCK:
    return test_move_open_lock(fd, cmd, arg);
  case F_TEST_MOVE_REC_LOCK:
    return test_move_rec_lock(fd, cmd, arg);
  case F_TEST_CLAMP_LOCKS:
    return test_clamp_locks(fd, arg);
  case F_TEST_UNCLAMP_LOCKS:
    return test_clamp_locks(fd, arg+2);
#endif

  default:
    return (*old_sys_fcntl)(fd, cmd, arg);
  }
}

int newsys_fcntl64(unsigned int fd, unsigned int cmd, unsigned long arg)
{
  switch (cmd & 0xff) {
  case F_OPLKREG:
  case F_OPLKREG_LVL2:
  case F_OPLKACK:
  case F_OPLKSTAT:
  case F_SET_SHMODE:
  case F_DEL_SHMODE:
  case F_SHARE_MODE:
  case F_EXCL_OPEN:
  case F_SHARED_OPEN:
    return oplock_fcntl(fd, cmd, arg);
  case F_SET_NLM_MODE:
  case F_DEL_NLM_MODE:
    return nlm_mode(cmd, arg);
  case F_SET_NLM_MOVE:
    return nlm_move(cmd, arg, fd);

#ifdef TEST_LOCK_MOVE_CLAMP
  case F_TEST_MOVE_OPEN_LOCK:
    return test_move_open_lock(fd, cmd, arg);
  case F_TEST_MOVE_REC_LOCK:
    return test_move_rec_lock(fd, cmd, arg);
  case F_TEST_CLAMP_LOCKS:
    return test_clamp_locks(fd, true);
  case F_TEST_UNCLAMP_LOCKS:
    return test_clamp_locks(fd, false);
#endif

  default:
    return (*old_sys_fcntl64)(fd, cmd, arg);
  }
}


#ifndef __SYSFUN_DEFS
#define __SYSFUN_DEFS
extern long sys_call_table[];
typedef asmlinkage long (*sysfun_long_p)();
typedef asmlinkage int (*sysfun_int_p)();
#endif

void oplock_init(void)
{
  TRACE0(TRACE_SMB, 5, TRCID_OPLOCK_INIT,
         "Loading kernel oplock support\n");
  old_sys_fcntl = (sysfun_long_p)sys_call_table[__NR_fcntl];
  old_sys_fcntl64 = (sysfun_long_p)sys_call_table[__NR_fcntl64];
  sys_call_table[__NR_fcntl] = (long)&newsys_fcntl;
  sys_call_table[__NR_fcntl64] = (long)&newsys_fcntl64;
}

void oplock_cleanup(void)
{
  sys_call_table[__NR_fcntl] = (long)old_sys_fcntl;
  sys_call_table[__NR_fcntl64] = (long)old_sys_fcntl64;
}
